package com.aliyun.auth.credentials.provider;

import com.aliyun.auth.credentials.ICredential;
import com.aliyun.core.utils.StringUtils;

import java.util.*;

public class DefaultCredentialProvider implements ICredentialProvider {
    //    private static final DefaultCredentialProvider DEFAULT_CREDENTIALS_PROVIDER = new DefaultCredentialProvider(builder());
    private List<ICredentialProvider> customizeProviders;
    private final CredentialsProviderChain providerChain;

    private DefaultCredentialProvider(Builder builder) {
        this.customizeProviders = builder.customizeProviders;
        this.providerChain = createChain(builder);
    }

//    public static DefaultCredentialProvider create() {
//        return DEFAULT_CREDENTIALS_PROVIDER;
//    }

    private CredentialsProviderChain createChain(Builder builder) {
        boolean asyncCredentialUpdateEnabled = builder.asyncCredentialUpdateEnabled;
        boolean reuseLastProviderEnabled = builder.reuseLastProviderEnabled;
        if (customizeProviders.isEmpty()) {
            ICredentialProvider[] credentialsProviders = new ICredentialProvider[]{
                    SystemPropertiesCredentialProvider.create(),
                    EnvironmentVariableCredentialProvider.create(),
                    ProfileCredentialProvider.builder()
                            .profileFile(builder.profileFile)
                            .clientType(builder.clientType)
                            .asyncCredentialUpdateEnabled(asyncCredentialUpdateEnabled)
                            .build()
            };
            customizeProviders = Arrays.asList(credentialsProviders);
        }
        return CredentialsProviderChain.builder()
                .reuseLastProviderEnabled(reuseLastProviderEnabled)
                .credentialsProviders(customizeProviders)
                .build();

    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public ICredential getCredentials() {
        return providerChain.getCredentials();
    }

    public Boolean containsCredentialProvider(ICredentialProvider provider) {
        return customizeProviders.contains(provider);
    }

    @Override
    public void close() {
        providerChain.close();
    }

    @Override
    public String toString() {
        Map<String, Object> fieldMap = new HashMap<>();
        fieldMap.put("providerChain", providerChain);
        return StringUtils.toAliString("DefaultCredentialsProvider", fieldMap);
    }

    public static final class Builder {
        private String profileFile;
        private String clientType;
        private Boolean reuseLastProviderEnabled = true;
        private Boolean asyncCredentialUpdateEnabled = false;
        private List<ICredentialProvider> customizeProviders = new ArrayList<ICredentialProvider>();

        public Builder profileFile(String profileFile) {
            this.profileFile = profileFile;
            return this;
        }

        public Builder clientType(String clientType) {
            this.clientType = clientType;
            return this;
        }

        public Builder reuseLastProviderEnabled(Boolean reuseLastProviderEnabled) {
            this.reuseLastProviderEnabled = reuseLastProviderEnabled;
            return this;
        }

        public Builder asyncCredentialUpdateEnabled(Boolean asyncCredentialUpdateEnabled) {
            this.asyncCredentialUpdateEnabled = asyncCredentialUpdateEnabled;
            return this;
        }

        public Builder customizeProviders(Collection<? extends ICredentialProvider> customizeProviders) {
            this.customizeProviders = new ArrayList<>(customizeProviders);
            return this;
        }

        public Builder customizeProviders(ICredentialProvider... customizeProviders) {
            return customizeProviders(Arrays.asList(customizeProviders));
        }

        public Builder addCustomizeProviders(ICredentialProvider credentialsProviders) {
            this.customizeProviders.add(credentialsProviders);
            return this;
        }

        public DefaultCredentialProvider build() {
            return new DefaultCredentialProvider(this);
        }
    }
}
