package com.alibaba.ttl.threadpool.agent.internal.transformlet.impl;

import com.alibaba.ttl.TtlCallable;
import com.alibaba.ttl.TtlRunnable;
import com.alibaba.ttl.spi.TtlAttachments;
import com.alibaba.ttl.spi.TtlEnhanced;
import com.alibaba.ttl.threadpool.agent.internal.logging.Logger;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import com.alibaba.ttl.threadpool.agent.internal.javassist.*;

import java.lang.reflect.Modifier;
import java.util.concurrent.Callable;

import static com.alibaba.ttl.TransmittableThreadLocal.Transmitter.capture;

/**
 * <b>Internal</b> utils for {@code Transformlet}.
 *
 * @author Jerry Lee (oldratlee at gmail dot com)
 * @since 2.6.0
 */
public class Utils {
    private static final Logger logger = Logger.getLogger(Utils.class);

    /**
     * String like {@code public ScheduledFuture scheduleAtFixedRate(Runnable, long, long, TimeUnit)}
     * for {@link  java.util.concurrent.ScheduledThreadPoolExecutor#scheduleAtFixedRate}.
     *
     * @param method method object
     * @return method signature string
     */
    @NonNull
    static String signatureOfMethod(@NonNull final CtBehavior method) throws NotFoundException {
        final StringBuilder stringBuilder = new StringBuilder();

        stringBuilder.append(Modifier.toString(method.getModifiers()));
        if (method instanceof CtMethod) {
            final String returnType = ((CtMethod) method).getReturnType().getSimpleName();
            stringBuilder.append(" ").append(returnType);
        }
        stringBuilder.append(" ").append(method.getName()).append("(");

        final CtClass[] parameterTypes = method.getParameterTypes();
        for (int i = 0; i < parameterTypes.length; i++) {
            CtClass parameterType = parameterTypes[i];
            if (i != 0) stringBuilder.append(", ");
            stringBuilder.append(parameterType.getSimpleName());
        }

        stringBuilder.append(")");
        return stringBuilder.toString();
    }

    @NonNull
    static String renamedMethodNameByTtl(@NonNull CtMethod method) {
        return "original$" + method.getName() + "$method$renamed$by$ttl";
    }

    static void doTryFinallyForMethod(@NonNull CtMethod method, @NonNull String beforeCode, @NonNull String finallyCode) throws CannotCompileException, NotFoundException {
        doTryFinallyForMethod(method, renamedMethodNameByTtl(method), beforeCode, finallyCode);
    }

    static void doTryFinallyForMethod(@NonNull CtMethod method, @NonNull String renamedMethodName, @NonNull String beforeCode, @NonNull String finallyCode) throws CannotCompileException, NotFoundException {
        final CtClass clazz = method.getDeclaringClass();
        final CtMethod newMethod = CtNewMethod.copy(method, clazz, null);

        // rename original method, and set to private method(avoid reflect out renamed method unexpectedly)
        method.setName(renamedMethodName);
        method.setModifiers(method.getModifiers()
                & ~Modifier.PUBLIC /* remove public */
                & ~Modifier.PROTECTED /* remove protected */
                | Modifier.PRIVATE /* add private */);

        final String returnOp;
        if (method.getReturnType() == CtClass.voidType) {
            returnOp = "";
        } else {
            returnOp = "return ";
        }
        // set new method implementation
        final String code = "{\n" +
                beforeCode + "\n" +
                "try {\n" +
                "    " + returnOp + renamedMethodName + "($$);\n" +
                "} finally {\n" +
                "    " + finallyCode + "\n" +
                "} }";
        newMethod.setBody(code);
        clazz.addMethod(newMethod);
        logger.info("insert code around method " + signatureOfMethod(newMethod) + " of class " + clazz.getName() + ": " + code);
    }

    @Nullable
    public static Object doCaptureWhenNotTtlEnhanced(@Nullable Object obj) {
        if (obj instanceof TtlEnhanced) return null;
        else return capture();
    }

    @Nullable
    public static Runnable doAutoWrap(@Nullable final Runnable runnable) {
        if (runnable == null) return null;

        final TtlRunnable ret = TtlRunnable.get(runnable, false, true);

        // have been auto wrapped?
        if (ret != runnable) setAutoWrapperAttachment(ret);

        return ret;
    }

    @Nullable
    public static <T> Callable<T> doAutoWrap(@Nullable final Callable<T> callable) {
        if (callable == null) return null;

        final TtlCallable<T> ret = TtlCallable.get(callable, false, true);

        // have been auto wrapped?
        if (ret != callable) setAutoWrapperAttachment(ret);

        return ret;
    }

    private static void setAutoWrapperAttachment(@Nullable final Object ttlAttachment) {
        if (!(ttlAttachment instanceof TtlAttachments)) return;

        ((TtlAttachments) ttlAttachment).setTtlAttachment(TtlAttachments.KEY_IS_AUTO_WRAPPER, true);
    }

    @Nullable
    public static Runnable doUnwrapIfIsAutoWrapper(@Nullable final Runnable runnable) {
        if (!(runnable instanceof TtlAttachments)) return runnable;

        // is an auto wrapper?
        final Boolean isAutoWrapper = ((TtlAttachments) runnable).getTtlAttachment(TtlAttachments.KEY_IS_AUTO_WRAPPER);
        if (!Boolean.TRUE.equals(isAutoWrapper)) return runnable;

        return TtlRunnable.unwrap(runnable);
    }

    @NonNull
    public static String getPackageName(@NonNull String className) {
        final int idx = className.lastIndexOf('.');
        if (-1 == idx) return "";

        return className.substring(0, idx);
    }

    public static boolean isClassAtPackage(@NonNull String className, @NonNull String packageName) {
        return packageName.equals(getPackageName(className));
    }

    public static boolean isClassUnderPackage(@NonNull String className, @NonNull String packageName) {
        String packageOfClass = getPackageName(className);
        return packageOfClass.equals(packageName) || packageOfClass.startsWith(packageName + ".");
    }

    public static boolean isClassAtPackageJavaUtil(@NonNull String className) {
        return isClassAtPackage(className, "java.util");
    }
}
