package com.aliyun.auth.credentials.provider;

import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.exception.*;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.utils.IOUtils;
import com.aliyun.core.utils.SdkAutoCloseable;
import com.aliyun.core.utils.StringUtils;
import com.aliyun.core.utils.Validate;

import java.util.*;

public class CredentialsProviderChain implements ICredentialProvider, SdkAutoCloseable {
    private static final ClientLogger logger = new ClientLogger(CredentialsProviderChain.class);
    private final List<ICredentialProvider> credentialsProviders;
    private final boolean reuseLastProviderEnabled;
    private volatile ICredentialProvider lastUsedProvider;

    private CredentialsProviderChain(Builder builder) {
        this.reuseLastProviderEnabled = builder.reuseLastProviderEnabled;
        this.credentialsProviders = Collections.unmodifiableList(
                Validate.notEmpty(builder.credentialsProviders, "No credential provider."));
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public ICredential getCredentials() throws CredentialException {
        if (reuseLastProviderEnabled && lastUsedProvider != null){
            return lastUsedProvider.getCredentials();
        }
        for (ICredentialProvider provider : credentialsProviders) {
            try {
                ICredential credentials = provider.getCredentials();
                logger.verbose("Loading credentials from " + provider);

                lastUsedProvider = provider;
                return credentials;
            } catch (RuntimeException e) {
                String message = provider + ": " + e.getMessage();
                logger.verbose("Unable to load credentials from " + message);
            }
        }
        throw new CredentialException("Unable to load credentials from any of the providers in the chain " + this);
    }

    @Override
    public void close() {
        credentialsProviders.forEach(credentials -> IOUtils.closeIfCloseable(credentials, logger));
    }

    @Override
    public String toString() {
        Map<String, Object> fieldMap = new HashMap<>();
        fieldMap.put("credentialsProviders", credentialsProviders);
        return StringUtils.toAliString("CredentialsProviderChain", fieldMap);
    }

    public static final class Builder {
        private List<ICredentialProvider> credentialsProviders = new ArrayList<>();
        private boolean reuseLastProviderEnabled = true;

        public Builder credentialsProviders(Collection<? extends ICredentialProvider> credentialsProviders) {
            this.credentialsProviders = new ArrayList<>(credentialsProviders);
            return this;
        }

        public Builder credentialsProviders(ICredentialProvider... credentialsProviders) {
            return credentialsProviders(Arrays.asList(credentialsProviders));
        }

        public Builder reuseLastProviderEnabled(Boolean reuseLastProviderEnabled) {
            this.reuseLastProviderEnabled = reuseLastProviderEnabled;
            return this;
        }

        public Builder addCredentialsProvider(ICredentialProvider credentialsProviders) {
            this.credentialsProviders.add(credentialsProviders);
            return this;
        }

        public CredentialsProviderChain build() {
            return new CredentialsProviderChain(this);
        }
    }
}
